# GeoJSON 파일 병합 및 QGIS DB 연동 가이드

이 문서는 **여러 개의 작은 GeoJSON 파일을 하나의 파일로 병합**하고, **QGIS를 이용하여 데이터베이스(PostGIS 등)에 효율적으로 연동**하는 절차를 안내합니다.

---

## 기존 방식의 문제점

| 도구          | 애로사항                                                                                                                                                    |
| ----------- | ------------------------------------------------------------------------------------------------------------------------------------------------------- |
| **ogr2ogr** | 스크립트 실행 중단 (서버 에러): ogr2ogr를 이용한 스크립트가 예상치 못한 이유로 중단되거나, GeoJSON 파일의 구조적 문제로 인해 오류가 발생하여 전체 작업이 멈추는 경우가 있었습니다. 특히 파일 수가 많을수록 불안정성이 커져 효율적인 자동화가 어려웠습니다. |
| **QGIS**    | 단건 처리의 한계: QGIS의 기본 레이어 추가 기능은 개별 파일을 수동으로 불러오는 방식이라 수십, 수백 개의 파일을 처리하기에 비효율적입니다. 일괄 처리 기능이 있지만, 종종 복잡한 설정이나 예상치 못한 문제로 인해 쾌적하게 작동하지 않는 경우가 있습니다.       |

---

## 개선 방안

* **ogr2ogr 대신 파이썬 스크립트**를 이용하여 여러 GeoJSON 파일을 안정적으로 병합
* Feature 타입과 FeatureCollection 타입 모두 지원
* 병합된 파일을 QGIS로 불러와 **PostGIS 등 데이터베이스에 직접 업로드**

---

## 1. 파이썬을 이용한 GeoJSON 파일 병합

아래 파이썬 스크립트를 `merge.py`로 저장 후, **GeoJSON 파일들이 있는 동일 디렉터리**에서 실행합니다.

### 파일 구조 예시

```
.
├── merge.py
├── wrnArea.L1010000.geojson
├── wrnArea.L1010200.geojson
├── wrnArea.L1010300.geojson
└── ...
```

### `merge.py` 코드

```python
import os
import glob
import json

def merge_geojson_files(input_folder, output_file):
    merged_features = []
    
    # [수정 필요] 파일명 패턴 변경 가능
    file_list = glob.glob(os.path.join(input_folder, 'wrnArea*.geojson'))

    if not file_list:
        print(f"오류: '{input_folder}' 폴더에서 GeoJSON 파일이 발견되지 않았습니다.")
        return

    print(f"병합할 GeoJSON 파일 {len(file_list)}개를 찾았습니다.")

    for file_path in file_list:
        try:
            with open(file_path, 'r', encoding='utf-8') as f:
                data = json.load(f)
                file_type = data.get('type')
                if file_type == 'FeatureCollection':
                    merged_features.extend(data.get('features', []))
                elif file_type == 'Feature':
                    merged_features.append(data)
                else:
                    print(f"경고: '{os.path.basename(file_path)}' 지원되지 않는 유형({file_type}) → 건너뜀")
        except json.JSONDecodeError as e:
            print(f"오류: '{os.path.basename(file_path)}' JSON 형식 오류: {e}")

    merged_geojson = {
        'type': 'FeatureCollection',
        'features': merged_features
    }

    try:
        with open(output_file, 'w', encoding='utf-8') as f:
            json.dump(merged_geojson, f, indent=2, ensure_ascii=False)
        print(f"\n성공적으로 {len(merged_features)}개의 피처를 '{output_file}'에 병합했습니다.")
    except IOError as e:
        print(f"오류: 출력 파일 '{output_file}' 저장 실패: {e}")

input_directory = '.'
output_filename = 'merged.geojson'
merge_geojson_files(input_directory, output_filename)
```

---

## 2. QGIS를 이용한 데이터베이스 연동

1. **QGIS 실행 및 레이어 불러오기**

   * QGIS 실행 → `레이어 > 레이어 추가 > 벡터 레이어 추가`
   * `merged.geojson` 파일 선택하여 불러오기

2. **데이터베이스 연결**

   * QGIS 메뉴 → `데이터베이스` → PostgreSQL/PostGIS 연결 설정

3. **데이터 내보내기**

   * `merged.geojson` 레이어에서 **마우스 오른쪽 클릭 → 내보내기 → 객체를 다른 이름으로 저장**

4. **내보내기 설정**

   * **형식**: `PostgreSQL` (데이터베이스 직접 연동 시)
   * **파일 이름**: 데이터베이스 테이블명
   * **좌표계**: 필요 시 `EPSG:4326 (WGS 84)`
   * **업로드 옵션**: 활성화

5. **실행**

   * 확인 클릭 → QGIS가 GeoJSON 데이터를 데이터베이스 테이블로 생성 및 삽입

---

## 최종 결과

* 여러 GeoJSON 파일이 **하나의 통합 테이블**로 데이터베이스에 저장
* QGIS 또는 다른 GIS 소프트웨어에서 테이블을 직접 불러와 활용 가능